using UnityEngine;
using System.Collections;

[System.Serializable]
public enum EasingMode
{
	None,
	InOutCubic,
	InOutQuintic,
	InQuintic,
	InQuartic,
	InCubic,
	InQuadratic,
	OutQuintic,
	OutQuartic,
	OutCubic,
	OutInCubic,
	OutInQuartic,
	BackInCubic,
	BackInQuartic,
	OutBackCubic,
	OutBackQuartic,
	OutElasticSmall,
	OutElasticBig,
	InElasticSmall,
	InElasticBig,
	Organic,
	InSine,
	OutSine,
	InOutSine,
	InElastic,
	OutElastic,
	InBack,
	OutBack
};

public class Ease
{
	public static float None (float t, float b, float c) { // time, b, distance,
		return b+c*(t);
	}
	
	public static float InOutCubic(float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (-2 * tc + 3 * ts);
	}
	
	public static float InOutQuintic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (6 * tc * ts + -15 * ts * ts + 10 * tc);
	}
	
	public static float InQuintic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (tc * ts);
	}
	
	public static float InQuartic (float t, float b, float c) {
		float ts = t * t;
		return b + c * (ts * ts);
	}
	
	public static float InCubic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (tc);
	}
	
	public static float InQuadratic (float t, float b, float c) {
		float ts = t * t;
		return b + c * (ts);
	}
	
	public static float OutQuintic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (tc * ts + -5 * ts * ts + 10 * tc + -10 * ts + 5 * t);
	}
	
	public static float OutQuartic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (-1 * ts * ts + 4 * tc + -6 * ts + 4 * t);
	}
	
	public static float OutCubic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (tc + -3 * ts + 3 * t);
	}
	
	public static float OutInCubic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (4 * tc + -6 * ts + 3 * t);
	}
	
	public static float OutInQuartic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (6 * tc + -9 * ts + 4 * t);
	}
	
	public static float BackInCubic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c *(4 * tc + -3 * ts);
	}
	
	public static float BackInQuartic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (2 * ts * ts + 2 * tc + -3 * ts);
	}
	
	public static float OutBackCubic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (4 * tc + -9 * ts + 6 * t);
	}
	
	public static float OutBackQuartic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (-2 * ts * ts + 10 * tc + -15 * ts + 8 * t);
	}
	
	public static float OutElasticSmall (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (33 * tc * ts + -106 * ts * ts + 126 * tc + -67 * ts + 15 * t);
	}
	
	public static float OutElasticBig (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b+c*(56*tc*ts + -175*ts*ts + 200*tc + -100*ts + 20*t);
	}
	
	public static float InElasticSmall (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (33 * tc * ts + -59 * ts * ts + 32 * tc + -5 * ts);
	}
	
	public static float InElasticBig (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (56 * tc * ts + -105 * ts * ts + 60 * tc + -10 * ts);
	}
	
	public static float Organic (float t, float b, float c) {
		float ts = t * t;
		float tc = ts * t;
		return b + c * (8.495f * tc * ts + -20.29f * ts * ts + 13.795f * tc + -1.7f * ts + 0.7f * t);		
	}
	
	public static float InSine (float t, float b, float c) {
		c -= b;
		return -c * Mathf.Cos(t / 1 * (Mathf.PI / 2)) + c + b;
	}

	public static float OutSine (float t, float b, float c) {
		c -= b;
		return c * Mathf.Sin(t / 1 * (Mathf.PI / 2)) + b;
	}
	
	public static float InOutSine (float t, float b, float c) {
		c -= b;
		return -c / 2 * (Mathf.Cos(Mathf.PI * t / 1) - 1) + b;
	}
	
	public static float InElastic (float t, float b, float c) {
		c -= b;
		
		float d = 1f;
		float p = d * .3f;
		float s = 0;
		float a = 0;
		
		if (t == 0) return b;
		
		if ((t /= d) == 1) return b + c;
		
		if (a == 0f || a < Mathf.Abs(c)){
			a = c;
			s = p / 4;
			}else{
			s = p / (2 * Mathf.PI) * Mathf.Asin(c / a);
		}
		
		return -(a * Mathf.Pow(2, 10 * (t-=1)) * Mathf.Sin((t * d - s) * (2 * Mathf.PI) / p)) + b;
	}	
	
	public static float OutElastic (float t, float b, float c) {
		c -= b;
		
		float d = 1f;
		float p = d * .3f;
		float s = 0;
		float a = 0;
		
		if (t == 0) return b;
		
		if ((t /= d) == 1) return b + c;
		
		if (a == 0f || a < Mathf.Abs(c)){
			a = c;
			s = p / 4;
			}else{
			s = p / (2 * Mathf.PI) * Mathf.Asin(c / a);
		}
		
		return (a * Mathf.Pow(2, -10 * t) * Mathf.Sin((t * d - s) * (2 * Mathf.PI) / p) + c + b);
	}
	
	public static float InBack(float t, float b, float c){
		c -= b;
		t /= 1;
		float s = 1.70158f;
		return c * (t) * t * ((s + 1) * t - s) + b;
	}

	
	public static float OutBack (float t, float b, float c) {
		float s = 1.70158f;
		c -= b;
		t = (t / 1) - 1;
		return c * ((t) * t * ((s + 1) * t + s) + 1) + b;
	}
	
	/*
	 * Returns eased version of the input float 
	 * */
	public static float Float(float t, EasingMode easingMode)
	{
		float easedT = 0;
		switch (easingMode) {
		case EasingMode.None:
			easedT = Ease.None(t, 0, 1);
			break;
		case EasingMode.InOutCubic:
			easedT = Ease.InOutCubic(t, 0, 1);
			break;
		case EasingMode.InOutQuintic:
			easedT = Ease.InOutQuintic(t, 0, 1);
			break;
		case EasingMode.InQuintic:
			easedT = Ease.InQuintic(t, 0, 1);
			break;
		case EasingMode.InQuartic:
			easedT = Ease.InQuartic(t, 0, 1);
			break;
		case EasingMode.InCubic:
			easedT = Ease.InCubic(t, 0, 1);
			break;
		case EasingMode.InQuadratic:
			easedT = Ease.InQuadratic(t, 0, 1);
			break;
		case EasingMode.OutQuintic:
			easedT = Ease.OutQuintic(t, 0, 1);
			break;
		case EasingMode.OutQuartic:
			easedT = Ease.OutQuartic(t, 0, 1);
			break;
		case EasingMode.OutCubic:
			easedT = Ease.OutCubic(t, 0, 1);
			break;
		case EasingMode.OutInCubic:
			easedT = Ease.OutInCubic(t, 0, 1);
			break;
		case EasingMode.OutInQuartic:
			easedT = Ease.OutInCubic(t, 0, 1);
			break;
		case EasingMode.BackInCubic:
			easedT = Ease.BackInCubic(t, 0, 1);
			break;
		case EasingMode.BackInQuartic:
			easedT = Ease.BackInQuartic(t, 0, 1);
			break;
		case EasingMode.OutBackCubic:
			easedT = Ease.OutBackCubic(t, 0, 1);
			break;
		case EasingMode.OutBackQuartic:
			easedT = Ease.OutBackQuartic(t, 0, 1);
			break;
		case EasingMode.OutElasticSmall:
			easedT = Ease.OutElasticSmall(t, 0, 1);
			break;
		case EasingMode.OutElasticBig:
			easedT = Ease.OutElasticBig(t, 0, 1);
			break;
		case EasingMode.InElasticSmall:
			easedT = Ease.InElasticSmall(t, 0, 1);
			break;
		case EasingMode.InElasticBig:
			easedT = Ease.InElasticBig(t, 0, 1);
			break;
		case EasingMode.Organic:
			easedT = Ease.Organic(t, 0, 1);
			break;
		case EasingMode.InSine:
			easedT = Ease.InSine(t, 0, 1);
			break;
		case EasingMode.OutSine:
			easedT = Ease.OutSine(t, 0, 1);
			break;
		case EasingMode.InOutSine:
			easedT = Ease.InOutSine(t, 0, 1);
			break;
		case EasingMode.InElastic:
			easedT = Ease.OutElastic(t, 0, 1);
			break;
		case EasingMode.OutElastic:
			easedT = Ease.OutElastic(t, 0, 1);
			break;
		case EasingMode.InBack:
			easedT = Ease.InBack(t, 0, 1);
			break;
		case EasingMode.OutBack:
			easedT = Ease.OutBack(t, 0, 1);
			break;
		default: easedT = 0;
			break;
		}
		
		return easedT;
	}
	
	public static Vector3 V3(Vector3 v1, Vector3 v2, float t, EasingMode easingMode) {
		float easedT = Ease.Float(t, easingMode);
		return ((1 - easedT) * v1) + (easedT * v2);
	}
	
	public static float LerpValue(float value, float target, float min, float max, float increaseSpeed, float decreaseSpeed) {
		if (value == target) return target;
		if (value < target) return Mathf.Clamp(value + Time.deltaTime * increaseSpeed, min, max);
		else return Mathf.Clamp(value - Time.deltaTime * decreaseSpeed, min, max);
	}
}